<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Folder;
use App\Models\Image;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class ImageController extends Controller
{
    public function index(Request $request)
    {
        $folders = Folder::all();
        $images = Image::whereNull('folder_id')->latest()->paginate(12);
        return view('admin.images.index', compact('folders', 'images'));
    }

    public function showFolder(Folder $folder)
    {
        $folders = Folder::all();
        $images = $folder->images()->paginate(12);
        return view('admin.images.show', compact('folder', 'images', 'folders'));
    }

    public function storeFolder(Request $request)
    {
        $request->validate(['name' => 'required|unique:folders,name']);

        $slug = Str::slug($request->name);

        // Create physical folder
        $dirPath = public_path('uploads/' . $slug);
        if (!File::exists($dirPath)) {
            File::makeDirectory($dirPath, 0755, true);
        }

        Folder::create([
            'name' => $request->name,
            'slug' => $slug
        ]);

        return back()->with('success', 'Folder created successfully');
    }

    public function updateFolder(Request $request, Folder $folder)
    {
        $request->validate(['name' => 'required|unique:folders,name,' . $folder->id]);

        $oldSlug = $folder->slug;
        $newSlug = Str::slug($request->name);

        if ($oldSlug !== $newSlug) {
            $oldPath = public_path('uploads/' . $oldSlug);
            $newPath = public_path('uploads/' . $newSlug);

            if (File::exists($oldPath)) {
                File::move($oldPath, $newPath);

                // Update all internal image paths in DB
                foreach ($folder->images as $image) {
                    $newImagePath = str_replace('uploads/' . $oldSlug, 'uploads/' . $newSlug, $image->path);
                    $image->update(['path' => $newImagePath]);
                }
            }
        }

        $folder->update([
            'name' => $request->name,
            'slug' => $newSlug
        ]);

        return back()->with('success', 'Folder updated successfully');
    }

    public function deleteFolder(Folder $folder)
    {
        // Delete physical folder
        $dirPath = public_path('uploads/' . $folder->slug);
        if (File::exists($dirPath)) {
            File::deleteDirectory($dirPath);
        }

        $folder->delete(); // Cascades to images in DB
        return redirect()->route('admin.images.index')->with('success', 'Folder deleted successfully');
    }

    public function uploadImages(Request $request)
    {
        $request->validate([
            'images.*' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'folder_id' => 'nullable|exists:folders,id',
            'custom_name' => 'nullable|string|max:255'
        ]);

        $folder = $request->folder_id ? Folder::find($request->folder_id) : null;
        $uploadSubDir = $folder ? $folder->slug : '';
        $uploadDir = public_path('uploads/' . $uploadSubDir);

        if (!File::exists($uploadDir)) {
            File::makeDirectory($uploadDir, 0755, true);
        }

        \Log::info('Upload attempt start', ['has_files' => $request->hasFile('images'), 'file_count' => count($request->file('images') ?? [])]);

        if ($request->hasFile('images')) {
            $files = $request->file('images');
            foreach ($files as $index => $file) {
                $ext = $file->getClientOriginalExtension();
                $originalName = $file->getClientOriginalName();

                if ($request->custom_name) {
                    $baseName = Str::slug($request->custom_name);
                    $filename = (count($files) > 1) ? $baseName . '_' . ($index + 1) . '.' . $ext : $baseName . '.' . $ext;

                    // Avoid collisions
                    if (File::exists($uploadDir . '/' . $filename)) {
                        $filename = time() . '_' . Str::random(5) . '_' . $filename;
                    }
                } else {
                    $filename = time() . '_' . Str::random(5) . '_' . $originalName;
                }

                $file->move($uploadDir, $filename);

                Image::create([
                    'folder_id' => $request->folder_id,
                    'filename' => $filename,
                    'original_name' => $originalName,
                    'path' => 'uploads/' . ($uploadSubDir ? $uploadSubDir . '/' : '') . $filename
                ]);
            }
        }

        return back()->with('success', 'Images uploaded successfully');
    }

    public function deleteImage(Image $image)
    {
        $this->deleteImageFile($image);
        $image->delete();
        return back()->with('success', 'Image deleted successfully');
    }

    public function moveImage(Request $request, Image $image)
    {
        $request->validate([
            'folder_id' => 'nullable|exists:folders,id'
        ]);

        $oldPath = public_path($image->path);

        $targetFolder = $request->folder_id ? Folder::find($request->folder_id) : null;
        $targetDir = $targetFolder ? $targetFolder->slug : '';
        $newDirFull = public_path('uploads/' . $targetDir);

        if (!File::exists($newDirFull)) {
            File::makeDirectory($newDirFull, 0755, true);
        }

        $newPathRelative = 'uploads/' . ($targetDir ? $targetDir . '/' : '') . $image->filename;
        $newPathFull = public_path($newPathRelative);

        if (File::exists($oldPath)) {
            File::move($oldPath, $newPathFull);
        }

        $image->update([
            'folder_id' => $request->folder_id,
            'path' => $newPathRelative
        ]);

        return back()->with('success', 'Image moved successfully');
    }

    public function getImagesAjax(Request $request)
    {
        $folderId = $request->folder_id;
        $folders = [];

        if ($folderId == 'root' || !$folderId) {
            $images = Image::whereNull('folder_id')->latest()->paginate(12);
            $folders = Folder::all();
        } else {
            $folder = Folder::findOrFail($folderId);
            $images = $folder->images()->latest()->paginate(12);
        }

        return response()->json([
            'images' => $images,
            'folders' => $folders,
            'current_folder' => $folderId
        ]);
    }

    private function deleteImageFile(Image $image)
    {
        $filePath = public_path($image->path);
        if (File::exists($filePath)) {
            File::delete($filePath);
        }
    }
}
