<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::with('image:id,path')->get(['id', 'name', 'slug', 'description', 'image_id']);

        return response()->json([
            'status' => 'success',
            'data' => $categories
        ]);
    }

    /** Build full URL for an image path. Uses APP_URL so images load from backend. */
    private function imageUrl(?string $path, Request $request): ?string
    {
        if (!$path || !is_string($path)) {
            return null;
        }
        $path = str_replace('\\', '/', trim($path, '/'));
        if ($path === '') {
            return null;
        }
        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));
        $base = rtrim(config('app.url'), '/');
        if ($base === '') {
            $base = rtrim($request->getSchemeAndHttpHost(), '/');
        }
        return $base . '/' . $encodedPath;
    }

    public function products(Request $request, $slug)
    {
        $category = Category::where('slug', $slug)->first();
        if (!$category) {
            return response()->json([
                'status' => 'error',
                'message' => 'Category not found'
            ]);
        }
        $products = Product::where('category_id', $category->id)
            ->with(['thumbnail:id,path', 'images'])
            ->select('id', 'category_id', 'title', 'slug', 'thumbnail_id', 'price', 'discount_price', 'short_description')
            ->paginate(10);

        $products->getCollection()->transform(function (Product $product) use ($request) {
            $thumbUrl = $this->imageUrl($product->thumbnail?->path, $request);
            $firstImage = $product->images->first();
            $firstUrl = $firstImage ? $this->imageUrl($firstImage->path, $request) : null;
            $product->setAttribute('thumbnail_url', $thumbUrl);
            $product->setAttribute('first_image_url', $firstUrl);
            $product->makeVisible(['thumbnail_url', 'first_image_url']);
            return $product;
        });

        return response()->json([
            'status' => 'success',
            'category' => $category->name,
            'data' => $products
        ]);
    }
}
